close all;

import casadi.*

%% -- General setup -- %%

% States (order: G, X, I, D)

x = SX.sym('x', 4);

% Control (order: dDdt, dist)

u = SX.sym('u', 2);

%% Initial values

G_init = 4.5;
X_init = 15.0;
I_init = 15.0;
D_init = 0.0;

x0 = vertcat(G_init, X_init, I_init, D_init);

% Parameters

P1 = 0.028735;
P2 = 0.028344;
P3 = 5.035e-5;
V1 = 12.0;
n = 5.0 / 54.0;

%% Insert the ODE from task 1 -->

dxdt = vertcat( ...
        -P1 * (x(1) - G_init) - (x(2) - X_init) * x(1) + u(2), ...
        -P2 * (x(2) - X_init) + P3 * (x(3) - I_init), ...
        -n * x(3) + x(4) / V1, ...
        u(1) ...
    );

% <--

%% Also, insert the time points and the number of intervals from task 1 -->

t0 = 0.0;
tf = 200.0;
N = 250;

% <--

% Time points

T = linspace(t0, tf, N + 1);

% Duration of a time interval

dt = (tf - t0) / N;

% Control data: the change rate for the insulin infusion is all zero,
% so since the initial value for D is also zero, we do not start to infuse
% insulin

u_icr_init = zeros(size(T(1:end-1))).';

% The values for the "nutrition disturbance" are created as follows

u_meal = 3.0 * exp(-0.05 * T(1:end-1)).';

% Stack the control's data together within one structure

u_init = horzcat(u_icr_init, u_meal);

% Define an dictionary with ODE dxdt, states x and "parameters" u
% (this means, u is to be constant over one integration interval, since it
% might only change between two integration steps)

ode = struct('x', x, 'p', u, 'ode', dxdt);

% Instantiate the CVDOES integrator that comes with CasADi, 
% and set the final time

cvodes_integrator = integrator('cvodes_integrator', 'cvodes', ode, ...
    struct('t0', 0.0, 'tf', dt));

% Run the integrator for each control interval, while initializing with the
% results of the former integration step

x_sim = {x0};

for k = 1:N 
    
    result_integration = cvodes_integrator('x0', x_sim{end}, 'p', u_init(k, :));
    x_sim = {x_sim{:} result_integration.xf};

end

% Rearrange the results data

x_sim = full(horzcat(x_sim{:}));

% Generate the multiple shooting equality constraints, and structs that contain
% the optimization variables, as well as their bounds and initial values

% Bounds for u_icr

u_icr_min = -inf;
u_icr_max = inf;

% States bounds

x_min = [0.0; 0.0; 0.0; 5.0];
x_max = [inf; inf; inf; 20.0];

% Introduce short expressions for the number of states and the number of controls

nx = x.numel();
nu = u.numel();

% Initialize a list for the optimization variables and its bounds and initials
% with the corresponding values of s0

V = {MX.sym('s1', nx)};

% Make sure the optimizer cannot choose the initial states

V_min = [x0];
V_max = [x0];
V_init = [x0];

% Initialize the objective with the first entry of eq. (12) for k = 0;
% during setup of the shooting constraints, further entries will be added to f

G_ref = 5.0;
D_ref = 13.0;

f = (V{end}(1) - G_ref)^2 + (V{end}(4) - D_ref)^2;

% Initialize lists for multiple shooting constraints and bounds

g = {};
g_min = [];
g_max = [];

for k = 1:N

    % Generation of the multiple shooting constraints:
    %
    % The solver needs a formulation of the following kind:
    %
    % 0 <= s(k+1) - r(s(k), q(k)) <= 0
    %
    % The solver used here generally demands inequality constraints, so
    % we need to define both g_min and g_max and set them to zero, so that the
    % inequality constraints constraining g to a value between 0 and 0 are
    % then in fact one equality constraint)

    % -->

    % Set the previous sk+1 to sk for the current interval

    sk = V{end};

    % Introduce an optimization variable for the current qk

    qk = MX.sym(['q',  num2str(k)], nu);

    % Introduce an optimization variable for the current sk+1

    skplus1 = MX.sym(['s', num2str(k+1)], nx);

    % Append to current multiple shooting constraints to g

    result_integration = cvodes_integrator('x0', sk, 'p', qk);
    g = {g{:} skplus1 - result_integration.xf};

    % < --

    g_min = [g_min; zeros(nx,1)];
    g_max = [g_max; zeros(nx,1)];

    % Collect the introduced optimization variables, as well as their
    % bounds and initials

    % Collect controls

    % -->

    q_min_k = [u_icr_min; u_meal(k)];
    q_max_k = [u_icr_max; u_meal(k)];
    q_init_k = [u_icr_init(k); u_meal(k)];

    % <--

    V = {V{:} qk};
    V_min = [V_min; q_min_k];
    V_max = [V_max; q_max_k];
    V_init = [V_init; q_init_k];
    
    % Collect states

    % -->

    s_min_k = x_min;
    s_max_k = x_max;
    s_init_k = x_sim(:,k);

    % <--

    V = {V{:} skplus1};
    V_min = [V_min; s_min_k];
    V_max = [V_max; s_max_k];
    V_init = [V_init; s_init_k];

    % Add the relevant entries of sk+1 to the objective

    % -->

    f = f + (V{end}(1) - G_ref)^2 + (V{end}(4) - D_ref)^2;

    % <--

end

% Vectorize collected variables

V = veccat(V{:});
g = veccat(g{:});

% Set up the NLP and the solver accordingly, and solve the optimization problem

nlp = struct('x', V, 'f', f, 'g', g);

nlpsolver = nlpsol('nlpsolver', 'ipopt', nlp);

solution = nlpsolver('x0', V_init, 'lbx', V_min, 'ubx', V_max, ...
    'lbg', g_min, 'ubg', g_max);

% Extract the optimization variables' values ("x") from the solution object,
% rearrange values according to V

V_opt = solution.x;

G_opt = full(V_opt(1:nx+nu:end));
X_opt = full(V_opt(2:nx+nu:end));
I_opt = full(V_opt(3:nx+nu:end));
D_opt = full(V_opt(4:nx+nu:end));

x_opt = full(horzcat(G_opt, X_opt, I_opt, D_opt));

u_icr_opt = full(V_opt(5:nx+nu:end));
u_opt = full(V_opt(6:nx+nu:end));

plot_glucose(T, x_opt, u_opt, u_icr_opt);
