import numpy as np
import matplotlib.pylab as plt
import casadi as ca
from PIL import Image
plt.close("all")

# load image
y = np.asarray(Image.open('puppy.png'))
nr = y.shape[0]             # number of rows
nc = y.shape[1]             # number of columns

#%% Formulate NLP in CasADi
# new instance of an optimization problem
opti = ca.Opti()

# Declare decision variable
x = opti.variable(nr, nc)

# build objective
# first term of objective, keep close to original image
f1 = ca.sum1(ca.sum2(np.sqrt((x - y)**2 + 1)))

# construct smoothing term
alpha = .5                                      # weight of smoothing term
drow = x[:-1,:] - x[1:,:]                       # difference between neighbours along rows
drow = ca.vertcat(drow, np.zeros((1, nc)))      # pad with zeros to get consistent dimensions
dcol = x[:,:-1] - x[:,1:]                       # difference between neighbours along columnns
dcol = ca.horzcat(dcol, np.zeros((nr, 1)))      # pad with zeros to get consistent dimensions
f2 = ca.sum1(ca.sum2(np.sqrt(drow**2 + dcol**2 + 1)))

# full objective
f = f1 + alpha * f2
# pass to casadi (f is an expression depending on decision variable x)
opti.minimize(f)         

MODE = 'exact'        # use exact Hessian (default)
# MODE = 'L-BFGS'         # use a limited memory BFGS Hessian approximation

opts = {}                                       # empty dictionary for options
if MODE == 'exact':
    # This constraint helps a lot for this case
    opti.subject_to(x[:] >= 0)

elif MODE == 'L-BFGS':
    opts['ipopt'] = {'hessian_approximation': 'limited-memory'}

# Set IPOPT as solver, pass options
opti.solver('ipopt', opts)
# initialize x to original image
opti.set_initial(x, y)
# Actually solve the problem
sol = opti.solve()

# read value of x at solution
xopt = sol.value(x)

# visualize
plt.figure()
plt.imshow(y, cmap='gray')
plt.title('original')

plt.figure()
plt.imshow(xopt, cmap='gray')
plt.title('smoothened')

im = Image.fromarray(xopt.astype('uint8'))
im.save("puppy_rec.png")