import numpy as np
import casadi as ca
import matplotlib.pylab as plt
plt.close("all")

# Declare problem parameters
N = 40                      # number of masses
m = 4/N                     # weight of each mass
D = (70 / 40) * N           # spring constant
g = 9.81                    # gravity

# declare new instance of Opti
opti = ca.Opti()

# declare decision variables
y = opti.variable(N)        # y-positions of masses
z = opti.variable(N)        # z-positions of masses

# construct objective
obj = 0
obj += m * g * ca.sum1(z)                                   # gravitational energy
obj += .5 * D * ca.sum1( ca.diff(y)**2 + ca.diff(z)**2 )    # spring potential energy
opti.minimize(obj)                                          # pass to casadi

# define constraints
# fix position of outermost masses
opti.subject_to(y[0] == -2)
opti.subject_to(z[0] == 1)
opti.subject_to(y[-1] == 2)
opti.subject_to(z[-1] == 1)

opti.solver('ipopt')                    # use IPOPT as solver

#%% Variation 1: Floor constraint
opti_1 = ca.Opti(opti)          # Create copy of opti
opti_1.subject_to(z >= 0)       # add floor constraint (inequality is element-wise)
sol = opti_1.solve()            # actually solve the problem

yopt = sol.value(y)             # obtain values at solution
zopt = sol.value(z)

# visualize
plt.figure(figsize=(10,4))
ylim = [-.4, 1.1]               # unify to use in both subplots

plt.subplot(121)
plt.title(r"Mit Bedingung $z_i \geq 0$")
plt.plot([-2, 2], [0, 0], 'k', label="Boden")
plt.plot(yopt, zopt, '.-', label="Loesung")
plt.legend(loc='upper center')
plt.xlabel("y")
plt.ylabel("z")
plt.ylim(ylim)


#%% Variation 2: Hill constraint
opti_2 = ca.Opti(opti)          # Create copy of opti
opti_2.subject_to(z >= -y**2)   # add hill constraint (inequality is element-wise)

sol1 = opti_2.solve()            # actually solve the problem

# let's try a different initialization
# variables not explicitly initialized are initialized at 0!
opti_2.set_initial(y, -1)
sol2 = opti_2.solve()

# visualize
plt.subplot(122)
plt.title(r"Mit Bedingung $z_i \geq - y_i^2$")

# draw hill constraint
ycon = np.linspace(-.6, .6, 50)
zcon = -ycon**2
plt.plot(ycon, zcon, 'k', label="Huegel")

# draw obtained solutions
plt.plot(sol1.value(y), sol1.value(z), '.-', label="Loesung 1")
plt.plot(sol2.value(y), sol2.value(z), '.-', label="Loesung 2")
plt.legend()
plt.xlabel("y")
plt.ylabel("z")
plt.ylim(ylim)

plt.tight_layout()
plt.savefig('chain.pdf')
