import casadi as ca
import matplotlib.pyplot as plt
import numpy as np

from ballistic_dynamics_RK4 import ballistic_dynamics_RK4
from ode import ode
from rk4step import rk4step


# for people who cannot see an interactive plot, uncomment the following lines
import matplotlib
if matplotlib.get_backend() == 'agg':
    matplotlib.use('WebAgg')
print(f'backend: {matplotlib.get_backend()}')


# parameters
vbar = 15             # max velocity
w0 = np.zeros((4,1))  # initial guess

# decision variables
v1 = ca.SX.sym('v1',2)
v2 = ca.SX.sym('v2',2)
w = ca.vertcat(v1, v2)

# objective
P = ballistic_dynamics_RK4(w)
p1 = P[0:2]
p2 = P[2:4]
dp = p1 - p2
f = dp.T @ dp

# constraints
g = []
lbg = []
ubg = []

# p1z >= 0
g.append(p1[1])
lbg.append(0)
ubg.append(ca.inf)

# p2z >= 0
g.append(p2[1])
lbg.append(0)
ubg.append(ca.inf)

# v1^2 <= vbar^2
g.append(v1.T@v1)
lbg.append(-ca.inf)
ubg.append(vbar**2)

# v2^2 <= vbar^2
g.append(v2.T@v2)
lbg.append(-ca.inf)
ubg.append(vbar**2)

nlp = {'x': w, 'f': f, 'g': ca.vertcat(*g)}

# Create IPOPT solver object
solver = ca.nlpsol('solver', 'ipopt', nlp)

# Solve the NLP
res = solver(
    x0=w0,                             # solution guess
    lbx=-ca.inf,                       # lower bound on x
    ubx=ca.inf,                        # upper bound on x
    lbg=ca.vertcat(*lbg),              # lower bound on g
    ubg=ca.vertcat(*ubg),              # upper bound on g
)


# simulate optimal solution
wsol = res["x"].full().flatten()
T = 0.5
M = 100
DT = T/M
X0 = np.array([0, 0, wsol[0], wsol[1], 10, 0, wsol[2], wsol[3]])

X = np.zeros((X0.size, M+1))
X[:, 0] = X0
for j in range(M):
    X[:,j+1] = rk4step(lambda t,x: ode(x), DT, X[:,j], 0).full().flatten()

plt.figure(1)
plt.plot(X[0,:], X[1,:], label=r'$p_1$')
plt.plot(X[4,:], X[5,:], label=r'$p_2$')
plt.legend()

plt.show()
