import casadi as ca
from matplotlib import cm
import matplotlib.pyplot as plt
import numpy as np


# for people who cannot see an interactive plot, uncomment the following lines
import matplotlib
if matplotlib.get_backend() == 'agg':
    matplotlib.use('WebAgg')
print(f'backend: {matplotlib.get_backend()}')


# Create NLP: Solve the Rosenbrock problem (unconstrained):
#     minimize    (x-1)^2 + 100*(y - x^2)^2

x = ca.SX.sym('x')
y = ca.SX.sym('y')
v = ca.vertcat(x, y)

def unc_rosenbrock_fun(x, y):
    return (1-x)**2 + 100*(y - x**2)**2
f = unc_rosenbrock_fun(x,y)

nlp = {'x': v, 'f': f}


# Create IPOPT solver object
solver = ca.nlpsol('solver', 'ipopt', nlp)

# Solve the NLP
res = solver(
    x0 = [2.5, 3.0],  # solution guess
    lbx = -ca.inf,  # lower bound on x
    ubx =  ca.inf,  # upper bound on g
)

 
# Print the solution
f_opt = res["f"].full()
print(f"f={f_opt}")
x_opt = res["x"].full()
print(f"x={x_opt}")
lam_x_opt = res["lam_x"].full()
print(f"lam_x={lam_x_opt}")
lam_g_opt = res["lam_g"].full()
print(f"lam_g={lam_g_opt}")


# Plot result
fig, ax = plt.subplots(subplot_kw={'projection': '3d'})
n_points = 50
range = 2

x_v = np.linspace(x_opt[0] - 2*range, x_opt[0] + range, n_points)
y_v = np.linspace(x_opt[1] - 2*range, x_opt[1] + 2*range, n_points)

[X, Y] = np.meshgrid(x_v, y_v)

ax.plot_surface(X, Y, unc_rosenbrock_fun(X,Y), cmap=cm.coolwarm)
ax.set_xlabel('x')
ax.set_ylabel('y')
ax.set_zlabel('z')

plt.show()
