import casadi as ca
from matplotlib import pyplot as plt
import numpy as np
from scipy.integrate import solve_ivp

from rk4step import rk4step


# for people who cannot see an interactive plot, uncomment the following lines
import matplotlib
if matplotlib.get_backend() == 'agg':
    matplotlib.use('WebAgg')
print(f'backend: {matplotlib.get_backend()}')


def dynamics(x):
    return ca.vertcat(x[1], -.2*x[1] - x[0])

Ts = .5
N = 19
x0 = np.array([1, 0])
t_grid = np.linspace(0, Ts*N, N+1)
t_span = [0, Ts*N]


# 2a)
# for task 2a, you need the following dynamics function because 
# solve_ivp requires a function with time as an argument, even 
# though in our case, the dynamics is not time dependant.
def dynamics_scipy(t, x):
    return dynamics(x).full().flatten()

sol = solve_ivp(dynamics_scipy, t_span, x0, t_eval=t_grid)
t_grid = sol.t
X_ode45 = sol.y

fig, ax = plt.subplots()
ax.set_xlabel('time t')
ax.set_ylabel('state x')

ax.plot(t_grid, X_ode45[0,:], 'kx-', label='x1 ode45')
ax.plot(t_grid, X_ode45[1,:], 'ko-', label='x2 ode45')


# 2b) euler
h = .125
N_sim = int(Ts*N/h)
t_eu = np.linspace(0, Ts*N, N_sim+1)
X_eu = np.zeros((x0.size, N_sim+1))
X_eu[:,0] = x0
for k in range(1, len(t_eu)):
    X_eu[:,k] = X_eu[:,k-1] + h * dynamics(X_eu[:, k-1]).full().flatten()

ax.plot(t_eu, X_eu[0,:], 'rx-', label='x1 euler')
ax.plot(t_eu, X_eu[1,:], 'ro-', label='x2 euler')


# 2b) rk4
X_rk4 = np.zeros((x0.size, N+1))
X_rk4[:,0] = x0
for k in range(1, len(t_grid)):
    X_rk4[:,k] = rk4step(dynamics, Ts,  X_rk4[:,k-1]).full().flatten()

ax.plot(t_grid, X_rk4[0,:], 'bx-', label='x1 rk4')
ax.plot(t_grid, X_rk4[1,:], 'bo-', label='x2 rk4')


# 2c) casadi
x = ca.MX.sym('x',2)
dt = ca.MX.sym('x',1)
x_next = ca.Function('x_next', [x,dt], [rk4step(dynamics, dt, x)])

X_ca = np.zeros((x0.size, N+1))
X_ca[:,0] = x0
for k in range(1, len(t_grid)):
    X_ca[:,k] = x_next(X_ca[:,k-1], Ts).full().flatten()


ax.plot(t_grid, X_ca[0,:], 'gx-', label='x1 casadi')
ax.plot(t_grid, X_ca[1,:], 'go-', label='x2 casadi')


ax.legend()
plt.show()
