close all;
clear;

%% Load measurements

data = load('volleyball_cameras.mat');

Y = data.Y;        % measurements, n_y x N              [1, ..., N]
X_true = data.X;   % true state trajectory, n_x x (N+1) [1, ..., N+1]

%% Parameter
N = size(Y, 2);

h = 0.05;            % step size 

std_w = 0.01;        % standard deviation of state noise
std_v = 5*10^(-4);   % standard deviation of measurement noise

e_l = [1 0 0]';          % line of sight
e_u = [0 0 1]';          % up vector
e_s = cross(e_u, e_l);   % right eye to left eye

d_eyes = 1;    % dist between eyes

L = 0.02;      % dist to screen

g = 9.81;      % gravity

P_R = [2  0.5 1.8]';        % position of right eye
P_L = P_R + d_eyes.*e_s;    % position of left eye

n_y = 4;                % two 2-dimensional cameras

n_p = 3;                % 3-dimensional position 
n_v = 3;                % 3-dimensional velocity

n_x = n_p + n_v;        % state dimension

% x_init_true = [5 -7 1.7 5 5 10]';  % true inital state

%% State model 
 
A = eye(n_x) + [zeros(n_p) h.*eye(n_v);
                zeros(n_p)  zeros(n_v)];

b = [0 0 -0.5*g*h^2 0 0 -h.*g]';

F = @(x) A*x + b;

%% Measurement model

alpha_R = @(p) L./((p-P_R)'*e_l); 
alpha_L = @(p) L./((p-P_L)'*e_l);

u_R = @(p)  e_u'*(alpha_R(p).*(p - P_R));
v_R = @(p) -e_s'*(alpha_R(p).*(p - P_R));

u_L = @(p)  e_u'*(alpha_L(p).*(p - P_L));
v_L = @(p) -e_s'*(alpha_L(p).*(p - P_L));

G = @(x) [v_R(x(1:3,1)) u_R(x(1:3,1)) v_L(x(1:3,1)) u_L(x(1:3,1))]';


%% Plot 3d trajectories

figure(1); hold on; grid on; axis equal;
plot3(X_true(1, :), X_true(2, :), X_true(3, :), 'kx-');

plot3(P_R(1), P_R(2), P_R(3), 'ro', 'MarkerSize', 5); 
plot3(P_L(1), P_L(2), P_L(3), 'bo', 'MarkerSize', 5);

quiver3(P_R(1), P_R(2), P_R(3), 5*e_l(1), 5*e_l(2), 5*e_l(3), 'r')
quiver3(P_L(1), P_L(2), P_L(3), 5*e_l(1), 5*e_l(2), 5*e_l(3), 'b')

view([-100, 7]);

%% Plot measurements

figure(2); 
s1 = subplot(2,1,2); hold on; grid on; 
title('Right eye');
plot(Y(1, :), Y(2,:), 'rx');
legend('Y measured');

s2= subplot(2,1,1); hold on; grid on;
title('Left eye');
plot(Y(3, :), Y(4,:), 'bx');
legend('Y measured');

linkaxes([s1, s2], 'x')
linkaxes([s1, s2], 'y')



