close all;
clear;

%% Parameter
N_max = 200;

h = 0.05;           % step size 

std_w = 0.01;        % standard deviation of state noise
std_v = 5*10^(-4);   % standard deviation of measurement noise

e_l = [1 0 0]';    % line of sight
e_u = [0 0 1]';

e_s = cross(e_u, e_l);   % right eye to left eye

d_eyes = 1;    % dist between eyes

L = 0.02;

g = 9.81;

P_R = [2  0.5 1.8]';        % position of right eye
P_L = P_R + d_eyes.*e_s;    % position of left eye

n_y = 4;                % two 2-dimensional cameras

n_p = 3;                % 3-dimensional position 
n_v = 3;                % 3-dimensional velocity

n_x = n_p + n_v;  

x_init = [5 -7 1.7 5 5 10]';  % inital state

%% State model 
 
A = eye(n_x) + [zeros(n_p) h.*eye(n_v);
                zeros(n_p)  zeros(n_v)];

b = [0 0 -0.5*g*h^2 0 0 -h.*g]';

F = @(x) A*x + b;

%% Measurement model

alpha_R = @(p) L./((p-P_R)'*e_l); 
alpha_L = @(p) L./((p-P_L)'*e_l);

u_R = @(p)  e_u'*(alpha_R(p).*(p - P_R));
v_R = @(p) -e_s'*(alpha_R(p).*(p - P_R));

u_L = @(p)  e_u'*(alpha_L(p).*(p - P_L));
v_L = @(p) -e_s'*(alpha_L(p).*(p - P_L));

G = @(x) [v_R(x(1:3,1)) u_R(x(1:3,1)) v_L(x(1:3,1)) u_L(x(1:3,1))]';

%% Data generation

X = zeros(n_x, N_max+1);
Y_true = zeros(n_y, N_max);

X(:, 1) = x_init;

N = 1;

while X(3, N) > 0 && N < N_max
    X(:, N+1) = F(X(:, N)) + std_w.*randn(n_x, 1);
    Y_true(:, N)   = G(X(:, N)); 
    
    N = N+1;
end

N = N-1;

X      = X(:, 1:(N+1));
Y_true = Y_true(:, 1:N);

Y = Y_true + std_v.*randn(n_y, N);

%% Residual function

R = @(X) residual(X, Y, F, G, N, n_x, n_y, std_w, std_v);

%% Initial guess: simulate system deterministically

X_init = zeros(n_x, N+1);
X_init(:, 1) = x_init + 1;   % disturb the true initial state

for n=1:N
    X_init(:, n+1) = F(X(:, n));
end

X_init = reshape(X_init, (N+1)*n_x, 1);    % reshape to column vector

%% Solve estimation problem

X_opt_col = lsqnonlin(R, X_init);
X_opt = reshape(X_opt_col, n_x, N+1);

R_opt = R(X_opt_col);

w_opt = R_opt(1:(N+1)*n_x, 1);
v_opt = R_opt((N+1)*n_x+1:end, 1);

disp(['Optimal cost = ', num2str(0.5.*R_opt'*R_opt)])

%% Plot 3d trajectories

figure(1); hold on; grid on;
plot3(X(1, :), X(2, :), X(3, :), 'bx-');

plot3(X_opt(1, :), X_opt(2, :), X_opt(3, :), 'r-');

plot3(P_R(1), P_R(2), P_R(3), 'ro', 'MarkerSize', 5); 
plot3(P_L(1), P_L(2), P_L(3), 'bo', 'MarkerSize', 5);

quiver3(P_R(1), P_R(2), P_R(3), 5*e_l(1), 5*e_l(2), 5*e_l(3))
quiver3(P_L(1), P_L(2), P_L(3), 5*e_l(1), 5*e_l(2), 5*e_l(3))

view([-100, 7]);

%% Plot 2d projections
figure(2); 
s1 = subplot(1,2,2); hold on; grid on;
title('Right eye');
plot(Y(1, :), Y(2,:), 'x-');
plot(Y_true(1, :), Y_true(2,:), '.-');
legend('Y measured', 'Y true');

s2= subplot(1,2,1); hold on; grid on;
title('Left eye');
plot(Y(3, :), Y(4,:), 'x-');
plot(Y_true(3, :), Y_true(4,:), '.-');
legend('Y measured', 'Y true');


linkaxes([s1, s2], 'x')
linkaxes([s1, s2], 'y')

save('volleyball_cameras.mat', 'Y', 'X'); 


%% Residual
function [R] = residual(X, Y, F, G, N, n_x, n_y, std_w, std_v)

    R = zeros(n_x*N + n_y*N, 1);

    x_next = X(1:n_x, 1);
    
    W = [(1./std_w).*eye(N*n_x)  zeros(N*n_x, N*n_y);
         zeros(N*n_y, N*n_x)    (1./std_v).*eye(N*n_y)];

    for n=1:N
        x_current = x_next;
        x_next = X((n*n_x+1):(n+1)*n_x, 1);

        i1 = (n-1)*n_x + 1;
        i2 = N*n_x + (n-1)*n_y + 1;

        R(i1:(i1+n_x-1), 1) = x_next - F(x_current); 
        R(i2:(i2+n_y-1), 1) = Y(:, n) - G(x_current);
    end
 
    R = W*R;
end



